package org.msh.tb.adminunits.population;

import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.msh.tb.entities.AdministrativeUnit;
import org.msh.tb.entities.Population;
import org.msh.utils.date.DateUtils;

import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by Mauricio on 08/05/2017.
 * Controller class to handle recording of population.
 *
 * Population will be informed yearly and by Upazila level.
 */
@Name("adminUnitPopulationHome")
@Scope(ScopeType.CONVERSATION)
public class AdminUnitPopulationHome {

    @In
    EntityManager entityManager;

    private Integer year;

    // the district id that is setting the upazillas population.
    private Integer parentId;
    private AdministrativeUnit parentAdminUnit;

    private List<AdminUnitPopulationWrapper> adminUnits;

    public void initialize() {
        if (year == null) {
            year = DateUtils.yearOf(DateUtils.getDate());
        }

        loadParentUnit();

        List<AdministrativeUnit> auList = entityManager.createQuery("from AdministrativeUnit au " +
                "where au.parent.id = :parentId " +
                "and au.countryStructure.level = 3 " + // has to be upazilla
                "order by au.name")
                .setParameter("parentId", parentId)
                .getResultList();

        adminUnits = new ArrayList();

        for (AdministrativeUnit adminUnit : auList) {
            Population population = findPopulation(adminUnit.getId());
            adminUnits.add(new AdminUnitPopulationWrapper(adminUnit, population == null ? null : population.getPopulation()));
        }
    }

    public String save() {

        if (getParentAdminUnit() == null || year == null) {
            return "error";
        }

        for (AdminUnitPopulationWrapper item : adminUnits) {
            Population population = findPopulation(item.getAdminUnit().getId());

            if (item.getPopulation() != null && item.getPopulation() > 0) {
                if (population == null) {
                    population = new Population();
                    population.setYear(year);
                    population.setAdminUnit(item.getAdminUnit());
                    population.setPopulation(item.getPopulation());
                    entityManager.persist(population);
                } else {
                    population.setPopulation(item.getPopulation());
                    entityManager.merge(population);
                }

                entityManager.flush();
            } else {
                if (population != null) {
                    entityManager.createQuery("delete from Population where id = :id")
                            .setParameter("id", population.getId())
                            .executeUpdate();
                }
            }
        }

        return "success";
    }

    private Population findPopulation(Integer adminUnitId) {
        Population population = null;

        try {
            population = (Population) entityManager.createQuery("from Population p " +
                    "where p.adminUnit.id = :auId " +
                    "and p.year = :year")
                    .setParameter("auId", adminUnitId)
                    .setParameter("year", year)
                    .getSingleResult();
        } catch (NoResultException e) {
            // do nothing
        }

        return population;
    }

    private void loadParentUnit() {
        if (parentId != null) {
            parentAdminUnit = entityManager.find(AdministrativeUnit.class, parentId);
        }
    }

    public EntityManager getEntityManager() {
        return entityManager;
    }

    public void setEntityManager(EntityManager entityManager) {
        this.entityManager = entityManager;
    }

    public List<AdminUnitPopulationWrapper> getAdminUnits() {
        return adminUnits;
    }

    public void setAdminUnits(List<AdminUnitPopulationWrapper> adminUnits) {
        this.adminUnits = adminUnits;
    }

    public Integer getYear() {
        return year;
    }

    public void setYear(Integer year) {
        this.year = year;
    }

    public Integer getParentId() {
        return parentId;
    }

    public void setParentId(Integer parentId) {
        this.parentId = parentId;
    }

    public AdministrativeUnit getParentAdminUnit() {
        if (parentAdminUnit == null) {
            loadParentUnit();
        }
        return parentAdminUnit;
    }

    public void setParentAdminUnit(AdministrativeUnit parentAdminUnit) {
        this.parentAdminUnit = parentAdminUnit;
    }
}
